using System;
using System.Collections.Generic;
using System.Text;
using SlikaLib;

namespace SkewDetection
{
    /*************************************************************************************
    /* 1. Za svaki piksel originalne slike izracuna se linija u Hough prostoru           *
    /* 2. Svaki piksel te linije se inkrementuje u Hough prostoru                        *
    /* 3. Na kraju pikseli na Hough slici sa najvecom vrijednoscu odgovaraju             *
    /*    setovima sa najvecim brojem kolinearnih piksela na originalu                   *
    /*************************************************************************************
    /*  Kod iz knjige, str 299, poglavlje 8: Optical Character Recognition               *
    /************************************************************************************/
    class Hough
    {
        private int omega;

        public int Omega
        {
            get { return omega; }
            set { omega = value; }
        }

        private int rmax;

        public int RMax
        {
            get { return rmax; }
            set { rmax = value; }
        }

        private int tmax;

        public int TMax
        {
            get { return tmax; }
            set { tmax = value; }
        }

        private float tmval;

        public float TMVal
        {
            get { return tmval; }
            set { tmval = value; }
        }

        private float[,] data;

        public float[,] Data
        {
            get { return data; }
            set { data = value; }
        }

        public void Transform(Slika original)
        {
            int centerX, centerY, r, w, i, j, nc, nr;
            long w2, h2;
            double conv = Math.PI / 180.0;

            nc = original.Width; //number of columns
            nr = original.Height; //number of rows

            //find center of picture (origin)
            centerX = nc / 2;
            centerY = nr / 2;

            w2 = nc * nc;
            h2 = nr * nr;

            //find max perpendicular distance from the origin to the line
            rmax = (int)(Math.Sqrt((double)(w2 + h2)) / 2.0);

            //create image for the Hough space - choose you own sampling
            data = new float[omega, 2 * rmax + 1];

            //set value of pixels of Hough image to zero
            for (r = 0; r < 2 * rmax + 1; r++)
                for (w = 0; w < omega; w++)
                    data[w, r] = 0;

            //set max theta and maximum pixel value of Hough image to zero
            tmax = 0; tmval = 0;

            //For each pixel in the original image, the line in Hough space
            //is computed, and each pixel ont that line in the Hough image is incremented.
            for (i = 0; i < nc; i++)
                for (j = 0; j < nr; j++)
                {
                    float pixVal = original.getPixelAt(i, j);
                    if (pixVal > 0)
                        for (w = 0; w < omega; w++)
                        {
                            r = (int)((i - centerX) * Math.Cos((double)(w * conv))
                                + (j - centerY) * Math.Sin((double)(w * conv)));
                            data[w, rmax + r] += 1;
                        }
                }

            //After all pixels have been processe, the pixels in the Hough image that 
            //have the largest values correspond to the largest number of colinear pixels 
            //in the original image.
            for (i = 0; i < omega; i++)
                for (j = 0; j < 2 * rmax + 1; j++)
                    if (data[i, j] > tmval)
                    {
                        tmval = data[i, j]; //max pixel value in hough space
                        tmax = i; //max theta
                    }
        }

        /* -------------------------------------------------------------------------------------------- */
        public Hough(Slika original)
        {
            omega = 180;
            Transform(original);
        }
    }
}